#!/usr/bin/env python3
"""
Enhanced Demo Script for Cooperative Command Game

Demonstrates the new features:
1. Hidden objectives that need to be discovered
2. Movement uncertainty based on team member roles
"""

import time
from env import CoopCommandEnv, GameConfig, GameDifficulty
from team_member import Position

def print_separator(title):
    print("\n" + "="*60)
    print(f" {title}")
    print("="*60)

def print_game_state(state):
    """Print formatted game state information"""
    print(f"Round: {state['current_round']}/{state['max_rounds']}")
    print(f"Score: {state['main_score']} (Normalized: {state['score_normalized']}/100)")
    
    # Show objectives (only visible ones)
    print(f"\nVisible Objectives ({len(state['mission_objectives'])}/{state['total_objectives_count']}):")
    for obj in state['mission_objectives']:
        status_marker = {
            'pending': '⏳',
            'in_progress': '🔄',
            'completed': '✅',
            'failed': '❌'
        }.get(obj['status'], '❓')
        
        print(f"  {status_marker} {obj['description']}")
    
    # Show hidden objectives count
    if state['hidden_objectives_count'] > 0:
        print(f"\n🔍 Hidden objectives remaining: {state['hidden_objectives_count']}")
    
    # Show discovery hints
    if state.get('discovery_hints'):
        print("\n💡 Discovery hints:")
        for hint in state['discovery_hints']:
            print(f"  {hint}")
    
    # Show team status
    print(f"\nTeam Status:")
    for member_id, member in state['team_status']['members'].items():
        pos = member['position']
        health_bar = "█" * (member['health'] // 10) + "░" * ((100 - member['health']) // 10)
        print(f"  {member['name']} ({member['role']}): Health {health_bar} {member['health']}% at ({pos['x']:.1f}, {pos['y']:.1f})")

def demo_hidden_objectives():
    """Demonstrate hidden objective discovery"""
    print_separator("HIDDEN OBJECTIVES DEMO")
    
    # Create medium difficulty game with more hidden objectives
    config = GameConfig(
        difficulty=GameDifficulty.MEDIUM,
        seed_index=0,
        enable_audio=False
    )
    
    env = CoopCommandEnv(config=config)
    state = env.start_game()
    
    print("🎯 Starting game with hidden objectives...")
    print_game_state(state)
    
    # Move scout around to discover hidden objectives
    print("\n🔍 Sending scout to explore and discover hidden objectives...")
    
    # Try different exploration areas
    exploration_commands = [
        ("move to position 20, 30", "Exploring northwest sector"),
        ("move to position 60, 40", "Exploring northeast sector"),
        ("move to position 40, 70", "Exploring southeast sector"),
        ("move to position 15, 80", "Exploring southwest sector"),
    ]
    
    for i, (command, description) in enumerate(exploration_commands):
        print(f"\n--- Round {i+1}: {description} ---")
        
        # Execute exploration command
        result = env.execute_command(command)
        print(f"Command result: {result['response']}")
        
        # Step the game
        state = env.step()
        
        # Check if any new objectives were discovered
        if state['hidden_objectives_count'] < len(env.game_logic.hidden_objectives) + len(state['mission_objectives']) - state['total_objectives_count']:
            print("🎉 New objective discovered!")
        
        print_game_state(state)
        
        # Stop if all objectives discovered or game ends
        if state['hidden_objectives_count'] == 0 or state['state'] != 'playing':
            break
    
    env.shutdown()

def demo_movement_uncertainty():
    """Demonstrate movement uncertainty"""
    print_separator("MOVEMENT UNCERTAINTY DEMO")
    
    config = GameConfig(
        difficulty=GameDifficulty.HARD,  # Higher uncertainty
        seed_index=1,
        enable_audio=False
    )
    
    env = CoopCommandEnv(config=config)
    state = env.start_game()
    
    print("🎯 Starting game to demonstrate movement uncertainty...")
    print_game_state(state)
    
    # Test movement with different roles
    test_moves = [
        ("move to position 50, 50", "Testing scout precision (should be most accurate)"),
        ("move to position 25, 25", "Testing heavy precision (should be least accurate)"),
        ("move to position 75, 75", "Testing medic precision (should be moderate)"),
    ]
    
    for i, (command, description) in enumerate(test_moves):
        print(f"\n--- {description} ---")
        
        # Get current team member positions before move
        current_state = env.get_game_state()
        team_members = list(current_state['team_status']['members'].items())
        
        if i < len(team_members):
            member_id, member_info = team_members[i]
            before_pos = member_info['position']
            print(f"Member: {member_info['name']} ({member_info['role']})")
            print(f"Position before: ({before_pos['x']:.1f}, {before_pos['y']:.1f})")
            
            # Execute move command with SPECIFIC target member
            result = env.execute_command(command, target_member=member_id)
            print(f"Command result: {result['response']}")
        else:
            # Fallback if not enough team members
            result = env.execute_command(command)
            print(f"Command result: {result['response']}")
        
        # Step the game
        state = env.step()
        
        # Show actual position after move
        if i < len(team_members):
            member_id, _ = team_members[i]
            after_member = state['team_status']['members'][member_id]
            after_pos = after_member['position']
            print(f"Position after: ({after_pos['x']:.1f}, {after_pos['y']:.1f})")
            
            # Calculate movement error
            if 'move to position' in command:
                target_coords = command.split('position ')[1].split(',')
                target_x, target_y = float(target_coords[0]), float(target_coords[1])
                error_distance = ((after_pos['x'] - target_x)**2 + (after_pos['y'] - target_y)**2)**0.5
                print(f"Movement error: {error_distance:.2f} units from target")
        
        time.sleep(1)  # Small delay for demonstration
    
    env.shutdown()

def demo_combined_features():
    """Demonstrate both features working together"""
    print_separator("COMBINED FEATURES DEMO")
    
    config = GameConfig(
        difficulty=GameDifficulty.HARD,
        seed_index=2,
        enable_audio=False
    )
    
    env = CoopCommandEnv(config=config)
    state = env.start_game()
    
    print("🎯 Demo: Hidden objectives + Movement uncertainty...")
    print_game_state(state)
    
    print("\n📋 Strategy: Use scout to explore (more accurate + better discovery)")
    
    # Execute a few strategic moves
    commands = [
        "move to position 30, 40",
        "move to position 60, 60", 
        "move to position 45, 25"
    ]
    
    for i, command in enumerate(commands, 1):
        print(f"\n--- Strategic Move {i} ---")
        result = env.execute_command(command)
        print(f"Result: {result['response']}")
        
        state = env.step()
        print_game_state(state)
        
        if state['state'] != 'playing':
            break
    
    env.shutdown()

def main():
    """Main demonstration function"""
    print("🎮 Enhanced Cooperative Command Game Demo")
    print("This demo showcases two new features:")
    print("1. Hidden Objectives - Some objectives are hidden and must be discovered")
    print("2. Movement Uncertainty - Team members have different movement precision")
    
    try:
        demo_hidden_objectives()
        time.sleep(2)
        
        demo_movement_uncertainty()
        time.sleep(2)
        
        demo_combined_features()
        
    except KeyboardInterrupt:
        print("\n\n⏹️  Demo interrupted by user")
    except Exception as e:
        print(f"\n\n❌ Demo error: {e}")
        import traceback
        traceback.print_exc()
    
    print("\n🎮 Demo completed!")

if __name__ == "__main__":
    main() 